﻿namespace Microsoft.Samples.PlanMyNight.Web.Infrastructure
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Text.RegularExpressions;
    using System.Web;

    /// <summary>
    /// A handler to combine js files.
    /// </summary>
    public class JSCombineHandler : IHttpHandler
    {
        /// <summary>
        /// Gets a value indicating whether a handler is reusable.
        /// </summary>
        public bool IsReusable
        {
            get { return false; }
        }

        /// <summary>
        /// Method that process the request.
        /// </summary>
        /// <param name="context">The current http context.</param>
        public void ProcessRequest(HttpContext context)
        {
            if (!string.IsNullOrEmpty(context.Request.QueryString["js"]))
            {
                string[] relativeFiles = context.Request.QueryString["js"].Split(',');
                List<string> absoluteFiles = new List<string>();

                var mergedJavascript = new System.Text.StringBuilder();

                for (int i = 0; i < relativeFiles.Length; i++)
                {
                    string file = relativeFiles[i];
                    if (file.EndsWith(".js", StringComparison.OrdinalIgnoreCase))
                    {
                        string absoluteFile = context.Server.MapPath(file);
                        if (File.Exists(absoluteFile))
                        {
                            mergedJavascript.Append(File.ReadAllText(absoluteFile));
                            absoluteFiles.Add(absoluteFile);
                        }
                    }
                }

                var compressedJavascript = Yahoo.Yui.Compressor.JavaScriptCompressor.Compress(mergedJavascript.ToString());

                using (var writer = new StreamWriter(context.Response.OutputStream))
                {
                    writer.Write(compressedJavascript);
                }

                SetHeaders(context, absoluteFiles.ToArray());
            }
        }

        /// <summary>
        /// Strips the whitespace from any .js file.
        /// </summary>
        /// <param name="body">The contents of the javascript file.</param>
        /// <returns>A string without white spaces.</returns>
        private static string StripWhitespace(string body)
        {
            string optimized = Regex.Replace(body, @"/\*.*?\*/\s*\n*", string.Empty, RegexOptions.Singleline);
            return Regex.Replace(optimized, @"^\s*//.*$\n?|^\s*$\n|^\s+|\s+$", string.Empty, RegexOptions.Multiline);
        }

        /// <summary>
        /// This will make the browser and server keep the output
        /// in its cache and thereby improve performance.
        /// </summary>
        /// <param name="context">The current http context.</param>
        /// <param name="files">An array of the file names that will be processed.</param>
        private static void SetHeaders(HttpContext context, string[] files)
        {
            context.Response.ContentType = "text/javascript";
            context.Response.AddFileDependencies(files);
            context.Response.Cache.VaryByParams["js"] = true;
            context.Response.Cache.SetETagFromFileDependencies();
            context.Response.Cache.SetLastModifiedFromFileDependencies();
            context.Response.Cache.SetValidUntilExpires(true);
            context.Response.Cache.SetExpires(DateTime.Now.AddDays(7));
            context.Response.Cache.SetCacheability(HttpCacheability.Public);
        }

        /// <summary>
        /// Writes the content of the individual javascript files to the response stream.
        /// </summary>
        /// <param name="context">The current http context.</param>
        /// <param name="file">The destination file to write the content.</param>
        private static void WriteContent(Stream outputStream, string file)
        {
            using (var writer = new StreamWriter(outputStream))
            {
                using (StreamReader reader = new StreamReader(file))
                {
                    writer.Write(reader.ReadToEnd());
                }
            }
        }
    }
}
